# Time-stamp: <fevcalc-func.r || Friday, March 20, 2009 || 10:54:37>


############################################################
###
### Cystic Fibrosis FEV & height percentile conversion tool
### 
### R implementation
### Version 2.11
### (c) Michal Kulich 2009
###
### Loads R coefficient datasets febtcoef_xx_yy.Rdata from current directory 
##
## Changes from 2.10:
##    Fixed minor bug in htperc()
## Changes from 1.11:
##    Added new equations calculated from 1994-2006 CF Registry data
##    Allowed choice of reference period
##    Changed nesting of functions and loading of coefficients
##    Fixed calling of max.col throughout
## Changes from 1.01:
##    None in this code.
## Changes from 0.91:
##    Includes function htperc() for quantiles of height given age.

fevperc <- function(fevL,quan,age,ht,sex,period=c("99_06","94_01","94_00","01_06"))
  ## Convert between FEV quantiles and FEV in liters
  ##
  ## INPUT:
  ##
  ##   fevL:       FEV1 in liters (vector)
  ##   quan:       FEV1 quantiles (scalar or vector; values 0.01-0.99)
  ##   age:        age values (vector; 6-40 years)
  ##   ht:         height values (vector; 105-190 cm [180 in women])
  ##   sex:        sex values (vector, same length as ht)
  ##               permissible codes: 1,'m','M','Male','male'
  ##                                  0,'f','F','Female','female'
  ##   period:     period of reference equation
  ##               permissible values: "94_01","99_06" (moving 8-year window)
  ##                                   "94_00","01_06" (early vs. late period)
  ##
  ## Provide either fevL or quan, not both
  ## Provide age or ht or both
  ## Each age has specific permissible ranges of height; if height falls
  ##      outside the range, NA is returned
  ## Age, ht, sex must have equal length. Scalars are replicated as needed.
  ##
  ##
  ## OUTPUT:
  ##
  ## fevL was not specified -> returns a matrix of fev [L]
  ##                           (rows=observations, columns=quantiles)
  ## quan was not specified -> returns a vector of FEV quantiles
  ##             If FEV falls below 0.01 quantile, 0 is returned.
  ##             If FEV falls above 0.99 quantile, 1 is returned
  ##
  ##
  ## Loads R coefficient datasets fevhtcoef_xx_yy.Rdata from current directory 
  ##
  {
    ## Check input
    if(missing(fevL)&&missing(quan))
      stop("Either FEV in L or quantile values must be given.")
    if(!missing(fevL)&&!missing(quan))
      stop("Both FEV in L and quantile values entered.")
    if(missing(age)&&missing(ht))
      stop("Age or height (or both) must be given.")
    sex.m <- is.element(sex, c('1','m','M','male','Male'))
    sex.f <- is.element(sex, c('0','f','F','female','Female'))
    if(any(!sex.m & !sex.f))
      stop("Unknown value in gender. Use 'm'/'f' or 1/0.")
    ## lengths <- c(length(sex),0,0)
    ## Fixed Feb. 19, 2003
    lengths <- c(length(sex),0,0,0)
    if(!missing(age)) lengths[2] <- length(age)
    if(!missing(ht)) lengths[3] <- length(ht)
    if(!missing(fevL)) lengths[4] <- length(fevL)    
    ml <- max(lengths[lengths>0])
    if(lengths[1]==1) sex.m <- rep(sex.m,ml)
    if(lengths[2])
      {
        if(any(age<6 | age>40)) stop('Age must be between 6 and 40')
        if(lengths[2]==1) age <- rep(age,ml)
        if(length(age)!=length(sex.m))
          stop('Incompatible lengths of age and sex')
      }
    if(lengths[3])
      {
        if(any(ht<105)) stop('Height must be at least 105 cm')
        if(lengths[3]==1) ht <- rep(ht,ml)
        if(length(ht)!=length(sex.m))
          stop('Incompatible lengths of height and sex')
        if(any(sex.m & ht>190))
          stop('Height in males must not exceed 190 cm')
        if(any(!sex.m & ht>180))
          stop('Height in females must not exceed 180 cm')
      }
    if(lengths[4] && lengths[4]!=length(sex.m))
          stop('Incompatible length of fevL')      
    if(!missing(quan) && any(quan<0.01 | quan>0.99))
      stop('Quantile must be between 0.01 and 0.99')

    ## Load coefficients for given period
    period <- match.arg(period)
    fname <- paste("fevhtcoef_",period,".RData",sep="")
    if(!file.exists(fname)) stop(paste("Could not find coefficients file",
                                       fname,"in directory",getwd()))
    load(file=fname)

    fevL.age <- function(perc,age,male,rnd=3)
      {
        fevL <- matrix(0,nrow=length(male),ncol=length(perc))
        if(any(male))
          fevL[male,] <- get.resp(perc,age[male],bases.mage,age.pts.m,
                                  uncoef.mage)
        if(any(!male))
          fevL[!male,] <- get.resp(perc,age[!male],bases.fage,age.pts.f,
                                   uncoef.fage)
        colnames(fevL) <- perc
        round(fevL,rnd)
      }
    
    
    fevL.ht <- function(perc,ht,male,rnd=3)
      {
        fevL <- matrix(0,nrow=length(male),ncol=length(perc))
        if(any(male))
          fevL[male,] <- get.resp(perc,ht[male],bases.mht,ht.pts.m,
                                  uncoef.mht)
        if(any(!male))
          fevL[!male,] <- get.resp(perc,ht[!male],bases.fht,ht.pts.f,
                                   uncoef.fht)
        colnames(fevL) <- perc
        round(fevL,rnd)
      }


    fevL.biv <- function(perc,age,ht,male,rnd=3)
      {
        fevL <- matrix(0,nrow=length(male),ncol=length(perc))
        if(any(male))
          fevL[male,] <- get.fev2(perc,age[male],ht[male],
                                  bases.mage,age.pts.m,bases.mht,ht.pts.m,
                                  bivcoef.m,fitint.m,htranges.m)
        if(any(!male))
          fevL[!male,] <- get.fev2(perc,age[!male],ht[!male],
                                   bases.fage,age.pts.f,bases.fht,ht.pts.f,
                                   bivcoef.f,fitint.f,htranges.f)
        colnames(fevL) <- perc
        round(fevL,rnd)
      }


    fevQ.age <- function(fev,age,male)
      ## calculate FEV quantile given FEV in liters, age, and sex
      ##
      ## Input:
      ##
      ## fev:        FEV1 in liters (vector)
      ## age:        age (vector)
      ## male:       sex (vector, male = 1, female = 0)
      ##
      ## Output: vector of FEV quantiles
      ##
      ## If FEV falls below 0.01 quantile, 0 is returned.
      ## If FEV falls above 0.99 quantile, 1 is returned
      ##
      ## This algorithm inverts fevL.age by a simple three-step method
      ## Faster and more efficient would be C or Fortran code
      ## Does not check for non-monotone FEV<->quantile association
      {
        ## determine first decimal place
        tryperc <- seq(0.1,0.9,by=0.1) 
        tryfev <- fevL.age(tryperc,age,male,4)
        fevint1 <- apply(matrix(fev,nrow=length(fev),ncol=9)>tryfev,1,sum)
        ## determine second decimal place
        fevint2 <- rep(0,length(fev))
        for(i in 0:9)
          {
            tryperc2 <- (i+tryperc)/10
            which <- ifelse(is.na(fevint1),F,fevint1==i)
            if(any(which))
              {
                tryfev <- fevL.age(tryperc2,age[which],male[which],4)
                fevint2[which] <-
                  apply(matrix(fev[which],nrow=sum(which),ncol=9)>tryfev,1,sum)
              }
          }
        ## determine third decimal place
        fevint3 <- rep(0,length(fev))
        tryperc <- seq(0,1,by=0.1) 
        for(i in 1:98)
          {
            tryperc3 <- (i+tryperc)/100
            which <- ifelse(is.na(fevint1),F,10*fevint1+fevint2==i)
            if(any(which))
              {
                tryfev <- fevL.age(tryperc3,age[which],male[which],4)
                ## this time, find the closest match
                fevint3[which] <- max.col(-abs(tryfev-fev[which]),ties.method="first")-1
                ## the values are 1-10
              }
          }
        ## put it all together
        fevq <- fevint1/10+fevint2/100+fevint3/1000
        ## subjects exceeding the 0.99 quantile get assigned to 1
        fevq <- ifelse(fevint1==9 & fevint2==9,1,fevq)
        round(fevq,3)
      }


    fevQ.ht <- function(fev,ht,male)
      ## calculate FEV quantile given FEV in liters, ht, and sex
      ##
      ## Input:
      ##
      ## fev:        FEV1 in liters (vector)
      ## ht:         height (vector)
      ## male:       sex (vector, male = 1, female = 0)
      ##
      ## Output: vector of FEV quantiles
      ##
      ## If FEV falls below 0.01 quantile, 0 is returned.
      ## If FEV falls above 0.99 quantile, 1 is returned
      ##
      ## This algorithm inverts fevL.ht by a simple three-step method
      ## Faster and more efficient would be C or Fortran code
      ## Does not check for non-monotone FEV<->quantile association
      {
        ## determine first decimal place
        tryperc <- seq(0.1,0.9,by=0.1) 
        tryfev <- fevL.ht(tryperc,ht,male,4)
        fevint1 <- apply(matrix(fev,nrow=length(fev),ncol=9)>tryfev,1,sum)
        ## determine second decimal place
        fevint2 <- rep(0,length(fev))
        for(i in 0:9)
          {
            tryperc2 <- (i+tryperc)/10
            which <- ifelse(is.na(fevint1),F,fevint1==i)
            if(any(which))
              {
                tryfev <- fevL.ht(tryperc2,ht[which],male[which],4)
                fevint2[which] <-
                  apply(matrix(fev[which],nrow=sum(which),ncol=9)>tryfev,1,sum)
              }
          }
        ## determine third decimal place
        fevint3 <- rep(0,length(fev))
        tryperc <- seq(0,1,by=0.1) 
        for(i in 1:98)
          {
            tryperc3 <- (i+tryperc)/100
            which <- ifelse(is.na(fevint1),F,10*fevint1+fevint2==i)
            if(any(which))
              {
                tryfev <- fevL.ht(tryperc3,ht[which],male[which],4)
                ## this time, find the closest match
                fevint3[which] <- max.col(-abs(tryfev-fev[which]),ties.method="first")-1
                ## the values are 1-10
              }
          }
        ## put it all together
        fevq <- fevint1/10+fevint2/100+fevint3/1000
        ## subjects exceeding the 0.99 quantile get assigned to 1
        fevq <- ifelse(fevint1==9 & fevint2==9,1,fevq)
        round(fevq,3)
      }
    

    fevQ.biv <- function(fev,age,ht,male)
      ## calculate FEV quantile given FEV in liters, age, height and sex
      ##
      ## Input:
      ##
      ## fev:        FEV1 in liters (vector)
      ## age:        age (vector)
      ## ht:         height (vector)
      ## male:       sex (vector, male = 1, female = 0)
      ##
      ## Output: vector of FEV quantiles
      ##
      ## If FEV falls below 0.01 quantile, 0 is returned.
      ## If FEV falls above 0.99 quantile, 1 is returned
      ##
      ## Each age has specific permissible ranges of height; if height falls
      ##      outside the range, empty value is returned
      ##
      ## This algorithm inverts fevL.biv by a simple three-step method
      ## Faster and more efficient would be C or Fortran code
      ## Does not check for non-monotone FEV<->quantile association
      {
        ## determine first decimal place
        tryperc <- seq(0.1,0.9,by=0.1) 
        tryfev <- fevL.biv(tryperc,age,ht,male,4)
        fevint1 <- apply(matrix(fev,nrow=length(fev),ncol=9)>tryfev,1,sum)
        ## determine second decimal place
        fevint2 <- rep(0,length(fev))
        for(i in 0:9)
          {
            tryperc2 <- (i+tryperc)/10
            which <- ifelse(is.na(fevint1),F,fevint1==i)
            if(any(which))
              {
                tryfev <- fevL.biv(tryperc2,age[which],ht[which],male[which],4)
                fevint2[which] <-
                  apply(matrix(fev[which],nrow=sum(which),ncol=9)>tryfev,1,sum)
              }
          }
        ## determine third decimal place
        fevint3 <- rep(0,length(fev))
        tryperc <- seq(0,1,by=0.1) 
        for(i in 1:98)
          {
            tryperc3 <- (i+tryperc)/100
            which <- ifelse(is.na(fevint1),F,10*fevint1+fevint2==i)
            if(any(which))
              {
                ## browser()
                tryfev <- fevL.biv(tryperc3,age[which],ht[which],male[which],4)
                ## this time, find the closest match
                ## max.col returns random results by default; use ties.method argument
                ## Changed Nov. 23, 2007
                fevint3[which] <- max.col(-abs(tryfev-fev[which]),ties.method="first")-1
              }
          }
        ## put it all together
        fevq <- fevint1/10+fevint2/100+fevint3/1000
        ## subjects exceeding the 0.99 quantile get assigned to 1
        fevq <- ifelse(fevint1==9 & fevint2==9,1,fevq)
        round(fevq,3)
      }
    
    
    
    ## select desired calculation
    if(!missing(age)&&!missing(ht))  ## bivariate
      {
        if(!missing(quan))           # quantiles -> liters
          out <- fevL.biv(quan,age,ht,sex.m)
        else                         # liters -> quantiles
          out <- fevQ.biv(fevL,age,ht,sex.m)
      }
    else
      {
        if(!missing(age))            ## age 
          {
            if(!missing(quan))       # quantiles -> liters
              out <- fevL.age(quan,age,sex.m)
            else                     # liters -> quantiles
              out <- fevQ.age(fevL,age,sex.m)
          }
        else                         ## height
          {
            if(!missing(quan))       # quantiles -> liters
              out <- fevL.ht(quan,ht,sex.m)
            else                     # liters -> quantiles
              out <- fevQ.ht(fevL,ht,sex.m)
          }
      }
    out
  }


htperc <- function(ht,quan,age,sex,period=c("99_06","94_01","94_00","01_06"))
  ## Convert between height quantiles and height in cm
  ##
  ## INPUT:
  ##
  ##   ht:         height values in cm (vector)
  ##   quan:       height quantiles (scalar or vector; values 0.01-0.99)
  ##   age:        age values (vector; 0-40 years)
  ##   sex:        sex values (vector, same length as ht)
  ##               permissible codes: 1,'m','M','Male','male'
  ##                                  0,'f','F','Female','female'
  ##   period:     period of reference equation
  ##               permissible values: "94_01","99_06" (moving 8-year window)
  ##                                   "94_00","01_06" (early vs. late period)
  ##
  ## Provide either ht or quan, not both
  ## Age, ht, sex must have equal length. Scalars are replicated as needed.
  ##
  ##
  ## OUTPUT:
  ##
  ## ht was not specified -> returns a matrix of heights [cm]
  ##                           (rows=observations, columns=quantiles)
  ## quan was not specified -> returns a vector of height quantiles
  ##             If height falls below 0.01 quantile, 0 is returned.
  ##             If height falls above 0.99 quantile, 1 is returned
  ##
  ## Loads R coefficient datasets fevhtcoef_xx_yy.Rdata from current directory 
  ##
  {
    ## Check input
    if(missing(ht)&&missing(quan))
      stop("Either height in cm or quantile values must be given.")
    if(!missing(ht)&&!missing(quan))
      stop("Both height in cm and quantile values entered.")
    if(missing(age))
      stop("Age must be given.")
    sex.m <- is.element(sex, c('1','m','M','male','Male'))
    sex.f <- is.element(sex, c('0','f','F','female','Female'))
    if(any(!sex.m & !sex.f))
      stop("Unknown value in gender. Use 'm'/'f' or 1/0.")
    lengths <- c(length(sex),length(age),0)
    if(!missing(ht)) lengths[3] <- length(ht)
    ml <- max(lengths[lengths>0])
    if(lengths[1]==1) sex.m <- rep(sex.m,ml)
    if(lengths[2])
      {
        if(any(age<0 | age>40)) stop('Age must be between 0 and 40')
        if(lengths[2]==1) age <- rep(age,ml)
        if(length(age)!=length(sex.m))
          stop('Incompatible lengths of age and sex')
      }
    if(lengths[3])
      {
        if(lengths[3]==1) ht <- rep(ht,ml)
        if(length(ht)!=length(sex.m))
          stop('Incompatible lengths of height and sex')
      }
    if(!missing(quan) && any(quan<0.01 | quan>0.99))
      stop('Quantile must be between 0.01 and 0.99')

    ## Load coefficients for given period
    period <- match.arg(period)
    fname <- paste("fevhtcoef_",period,".RData",sep="")
    if(!file.exists(fname)) stop(paste("Could not find coefficients file",
                                       fname,"in directory",getwd()))
    load(file=fname)

    htCm.age <- function(perc,age,male,rnd=1)
      {
        ht <- matrix(0,nrow=length(male),ncol=length(perc))
        if(any(male))
          ht[male,] <- get.resp(perc,age[male],bases.aht,age.pts.ht,
                                htqcoef.m)
        if(any(!male))
          ht[!male,] <- get.resp(perc,age[!male],bases.aht,age.pts.ht,
                                 htqcoef.f)
        colnames(ht) <- perc
        round(ht,rnd)
      }

    htQ.age <- function(ht,age,male)
      ## calculate height quantile given height in cm, age, and sex
      ##
      ## Input:
      ##
      ## ht:         height in cm (vector)
      ## age:        age (vector)
      ## male:       sex (vector, male = 1, female = 0)
      ##
      ## Output: vector of height quantiles
      ##
      ## If height falls below 0.01 quantile, 0 is returned.
      ## If height falls above 0.99 quantile, 1 is returned
      ##
      ## This algorithm inverts htCm.age by a simple three-step method
      ## Faster and more efficient would be C or Fortran code
      ## Does not check for non-monotone height<->quantile association
      {
        ## determine first decimal place
        tryperc <- seq(0.1,0.9,by=0.1) 
        tryht <- htCm.age(tryperc,age,male,2)
        htint1 <- apply(matrix(ht,nrow=length(ht),ncol=9)>tryht,1,sum)
        ## determine second decimal place
        htint2 <- rep(0,length(ht))
        for(i in 0:9)
          {
            tryperc2 <- (i+tryperc)/10
            which <- ifelse(is.na(htint1),F,htint1==i)
            if(any(which))
              {
                tryht <- htCm.age(tryperc2,age[which],male[which],2)
                htint2[which] <-
                  apply(matrix(ht[which],nrow=sum(which),ncol=9)>tryht,1,sum)
              }
          }
        ## determine third decimal place
        htint3 <- rep(0,length(ht))
        tryperc <- seq(0,1,by=0.1) 
        for(i in 1:98)
          {
            tryperc3 <- (i+tryperc)/100
            which <- ifelse(is.na(htint1),F,10*htint1+htint2==i)
            if(any(which))
              {
            tryht <- htCm.age(tryperc3,age[which],male[which],2)
            ## this time, find the closest match
            htint3[which] <- max.col(-abs(tryht-ht[which]),ties.method="first")-1
            ## the values are 1-10
          }
          }
        ## put it all together
        htq <- htint1/10+htint2/100+htint3/1000
        ## subjects exceeding the 0.99 quantile get assigned to 1
        htq <- ifelse(htint1==9 & htint2==9,1,htq)
        round(htq,3)
      }
    

    ## select desired calculation
    if(!missing(quan))           # quantiles -> liters
      out <- htCm.age(quan,age,sex.m)
    else                         # liters -> quantiles
      out <- htQ.age(ht,age,sex.m)
    out
  }






get.resp <- function(perc,x,bases.coef,int.pts,qr.coef)
  ## calculate response (FEV in liters, height in cm) given quantiles
  ## and an independent variable (age, height)
  ##
  ## interpolate linearly between available quantiles
  ##
  ## Input:
  ##
  ## perc:       desired quantiles (vector; 0.01 - 0.99)
  ## x:          independent variable values (vector)
  ## bases.coef: coefficient for spline bases (matrix)
  ## int.pts:    spline knots (incl. boundary knots) (vector)
  ## qr.coef:    coefficients from quantile regression models
  ##
  ## Output: matrix of responses (rows=observations, columns=quantiles)
  {
    nsub <- length(x)
    bases <- calc.base(x,bases.coef,int.pts)
    if(is.matrix(bases))
      mx <- cbind(rep(1,nsub),bases)
    else
      mx <- c(1,bases)
    ## interpolate between fitted percentiles
    ## find between which two of the 100 fitted percentiles
    ##    the values in perc lie
    lowperc <- floor(perc*100+0.001)
    upperc <- ceiling(perc*100-0.001)
    ## the adjustment by 0.001 above is necessary
    ## even in double precision, 0.28 is rendered as 0.28+3.5e-17
    ## this leads to trouble
    lowresp <- mx%*%t(qr.coef[lowperc,-1])
    upresp <- mx%*%t(qr.coef[upperc,-1])
    resp <- lowresp+(upresp-lowresp)*
      matrix(perc*100-lowperc,nrow=nsub,ncol=length(perc),byrow=T)
    resp
  }



get.fev2 <- function(perc,age,ht,bases.age,age.pts,bases.ht,ht.pts,
                     bivcoef,fitint,htranges)
  ## calculate FEV in liters given quantiles, age and height
  ##
  ## interpolate linearly between available FEV quantiles
  ##
  ## Input:
  ##
  ## perc:       desired quantiles (vector)
  ## age:        ages [yr] (vector)
  ## ht:         heights [cm] (vector)
  ## bases.age:  coefficients for age spline bases (matrix)
  ## age.pts:    age spline knots (incl. boundary knots) (vector)
  ## bases.ht:   coefficients for height spline bases (matrix)
  ## ht.pts:     height spline knots (incl. boundary knots) (vector)
  ## bivcoef:    coefficients from bivariate quantile regression
  ## fitint:     indicators of included interaction terms
  ## htranges:   tables of acceptable height ranges for each age
  ##             values outside the range get missing FEV
  ##
  ## Output: matrix of fev [L] (rows=observations, columns=quantiles)
  {
    nsub <- length(age)
    agebases <- calc.base(age,bases.age,age.pts)
    htbases <- calc.base(ht,bases.ht,ht.pts)
    ## find which height observations are within ranges
    agematch <- round(age,0)-5
    nranges <- dim(htranges)[1]
    agematch <- pmin(agematch,nranges)
    htin <- ht>=htranges$hlow[agematch] & ht<=htranges$hup[agematch]
    ## build model matrix
    nhtb <- ifelse(is.matrix(htbases),dim(htbases)[2],length(htbases))
    nageb <- ifelse(is.matrix(agebases),dim(agebases)[2],length(agebases))
    hind <- rep(1:nhtb,nageb)
    aind <- rep(1:nageb,rep(nhtb,nageb))
    if(is.matrix(htbases))
      {
        mx <- (htbases[,hind]*agebases[,aind])
        mx <- mx[,fitint==1]
        mx <- cbind(rep(1,nsub),htbases,agebases,mx)
      }
    else
      {
        mx <- (htbases[hind]*agebases[aind])
        mx <- mx[fitint==1]
        mx <- c(rep(1,nsub),htbases,agebases,mx)
      }
    ## interpolate between fitted percentiles
    ## find between which two of the 100 fitted percentiles
    ##    the values in perc lie
    lowperc <- floor(perc*100+0.001)
    upperc <- ceiling(perc*100-0.001)
    ## the adjustment by 0.001 above is necessary
    ## even in double precision, 0.28 is rendered as 0.28+3.5e-17
    ## this leads to trouble
    lowfev <- mx%*%t(bivcoef[lowperc,-1])
    upfev <- mx%*%t(bivcoef[upperc,-1])
    fev <- lowfev+(upfev-lowfev)*
      matrix(perc*100-lowperc,nrow=nsub,ncol=length(perc),byrow=T)
    fev[!htin] <- NA
    fev
  }


calc.base <- function(x,sbc,int.pts)
  ## An improved function to generate spline bases
  ## for a vector of observations x
  ## Coefficients are in data frame sbc, knots in vector int.pts
  {
    ## number of subjects
    nsub <- length(x)
    ## number of bases
    nb <- max(sbc$basis)
    ## number of intervals
    nint <- length(int.pts)-1
    ## assign interval numbers to x values
    xf <- as.numeric(unclass(cut(x,int.pts,include.lowest=T)))
    ## build matrix of coefficients for the x's
    rowinds <- rep(0:(nb-1),nsub)*nint+rep(xf,rep(nb,nsub))
    cxmat <- sbc[rowinds,3:10]
    ## multiply the coefficients by powers of x
    ## absolute term
    bases <- ifelse(cxmat$d0==0,0,cxmat$n0/cxmat$d0)
    xvec <- rep(x,rep(nb,nsub))
    xvec2 <- rep(x*x,rep(nb,nsub))
    ## linear term
    bases <- bases+ifelse(cxmat$d1==0,0,cxmat$n1*xvec/cxmat$d1)
    ## quadratic term
    bases <- bases+ifelse(cxmat$d2==0,0,cxmat$n2*xvec2/cxmat$d2)
    ## cubic term
    bases <- bases+ifelse(cxmat$d3==0,0,cxmat$n3*xvec2/cxmat$d3*xvec)
    ## reshape bases
    bases <- matrix(bases,nrow=nsub,ncol=nb,byrow=T)
    ## first column is removed (we have a separate intercept in the models)
    bases[,-1]
  }

